import struct
import inspect
import asyncio

from devicepilot.urpc.urpcendpoint import URPCEndpoint, float_to_q31, q31_to_float
from .eefmeasurementparameter import EEFMEASUREMENTPARAMETER_TYPE
from .eefmeasurementparameter import EEFMeasurementParameter


class EEFMeasurementFunctions(URPCEndpoint):

	def __init__(self, can_id: int, transmitter_queue, reception_queue_child, send, loop):
		super().__init__(can_id, transmitter_queue, reception_queue_child, send, loop)
		URPCEndpoint.CALLBACK_DIC[(can_id, 0)] = (self.send_interlock_event, '<B')
		self.__send_interlock_event_events = []
		URPCEndpoint.CALLBACK_DIC[(can_id, 1)] = (self.send_error_event, '<B')
		self.__send_error_event_events = []

	@staticmethod
	def get_own_interface_version():
		interface_major = 1
		interface_minor = 0
		return interface_major, interface_minor

	async def get_interface_version(self, type: int, timeout=1):
		"""
		<summary>
		Get the version of the interface currently used by the endpoint.
		</summary>
		
		Check that the returned type matches the requested type to ensure that the configuration is correct.
		If this interface is a derived interface and the requested type matches a base interface, than the type and
		version of the requested base interface are returned.
		
		<param name="type" out="false"> <b>Input:</b> The type of the interface. </param>
		<param name="pTypeCheck" out="true"> <b>Output:</b> The given type if supported. Otherwise the actual type of the interface. </param>
		<param name="pMajor" out="true"> <b>Output:</b> The major version of the interface. </param>
		<param name="pMinor" out="true"> <b>Output:</b> The minor version of the interface. </param>
		
		<returns>Immediately</returns>
		"""
		ticket, event = self.post_request(0, struct.pack('<H', type), '<HHH', False)
		val = await asyncio.wait_for(event, timeout)
		return val if len(val) > 1 else val[0]

	async def set_interface_version(self, type: int, major: int, minor: int, timeout=1):
		"""
		<summary>
		Set the type and version of the interface to be used by the endpoint.
		</summary>
		
		An endpoint can support multiple interface versions. If the current version of the interface is not compatible
		with the controller this function can be used to select another interface version. If an equal major version
		and greater or equal minor version is supported it will be used by the interface. If the requested type or
		version is not supported an error will be returned.
		
		<param name="type" out="false"> <b>Input:</b> The type of the interface. </param>
		<param name="major" out="false"> <b>Input:</b> The major version of the interface. </param>
		<param name="minor" out="false"> <b>Input:</b> The minor version of the interface. </param>
		
		<returns>Immediately</returns>
		
		<exception cref="URPCError::InterfaceTypeCheck"> The interface type is not supported by this endpoint. </exception>
		<exception cref="URPCError::InterfaceVersionCheck"> The requested interface version is not supported by this endpoint. </exception>
		"""
		ticket, event = self.post_request(1, struct.pack('<HHH', type, major, minor), '<', False)
		await asyncio.wait_for(event, timeout)

	async def reset(self, timeout=1):
		"""
		<summary>
		Stop any running sequence, reset all internal states, clear the sequence memory and the measurement results.
		</summary>
		
		<returns>Immediately</returns>
		"""
		ticket, event = self.post_request(2, None, '<', False)
		await asyncio.wait_for(event, timeout)

	async def get_status(self, timeout=1):
		"""
		<summary>
		Asks for the status.
		</summary>
		
		<param name="pStatus" out="true"> <b>Output:</b> The status value. </param>
		<param name="pError" out="true"> <b>Output:</b> 0 = No Error or an URPC Error Code </param>
		
		<returns>Immediately</returns>
		"""
		ticket, event = self.post_request(3, None, '<BH', False)
		val = await asyncio.wait_for(event, timeout)
		return val if len(val) > 1 else val[0]

	async def set_parameter(self, number: int, value: int, timeout=1):
		"""
		<summary>
		Sets a parameter.
		</summary>
		
		<param name="number" out="false"> <b>Input:</b> The number of the parameter. They are defined in MeasurementParameter.hpp. </param>
		<param name="value" out="false"> <b>Input:</b> The new value of the parameter. </param>
		
		<returns>Immediately</returns>
		
		<exception cref="MeasurementError::UnknownParameterNumber"> The given parameter number is not defined. </exception>
		"""
		ticket, event = self.post_request(4, struct.pack('<H'+EEFMEASUREMENTPARAMETER_TYPE[EEFMeasurementParameter(number)][0], number, value if len(EEFMEASUREMENTPARAMETER_TYPE[EEFMeasurementParameter(number)]) == 1 else float_to_q31(value)), '<', False)
		await asyncio.wait_for(event, timeout)

	async def get_parameter(self, number: int, timeout=1):
		"""
		<summary>
		Gets a parameter.
		</summary>
		
		<param name="number" out="false"> <b>Input:</b> The number of the parameter. They are defined in MeasurementParameter.hpp. </param>
		<param name="pValue" out="true"> <b>Output:</b> The value of the parameter. </param>
		
		<returns>Immediately</returns>
		
		<exception cref="MeasurementError::UnknownParameterNumber"> The given parameter number is not defined. </exception>
		"""
		ticket, event = self.post_request(5, struct.pack('<H', number), '<'+EEFMEASUREMENTPARAMETER_TYPE[EEFMeasurementParameter(number)][0], False)
		val = await asyncio.wait_for(event, timeout)
		return val[0] if len(EEFMEASUREMENTPARAMETER_TYPE[EEFMeasurementParameter(number)]) == 1 else q31_to_float(val[0])

	async def write_sequence(self, address: int, size: int, data: list, timeout=1):
		"""
		<summary>
		Write a new sequence or overwrite the existing sequence in the sequence memory.
		</summary>
		
		<param name="address" out="false"> <b>Input:</b> The start address of the sequence to write. </param>
		<param name="size" out="false"> <b>Input:</b> The number of instructions to write. </param>
		<param name="data" out="false" length="size"> <b>Input:</b> The instructions of the sequence. </param>
		
		<returns>Immediately</returns>
		"""
		ticket, event = self.post_request(6, struct.pack(f'<HB{size}L', address, size, *data), '<', False)
		await asyncio.wait_for(event, timeout)

	async def start_sequence(self, address: int, timeout=1):
		"""
		<summary>
		Start the sequence with the given number.
		</summary>
		
		<param name="address" out="false"> <b>Input:</b> The start address of the sequence. </param>
		
		<returns>Immediately</returns>
		"""
		ticket, event = self.post_request(7, struct.pack('<H', address), '<', False)
		await asyncio.wait_for(event, timeout)

	async def cancel_sequence(self, timeout=1):
		"""
		<summary>
		Cancel any running sequence and reset all internal states.
		</summary>
		
		<returns>Immediately</returns>
		"""
		ticket, event = self.post_request(8, None, '<', False)
		await asyncio.wait_for(event, timeout)

	async def read_results(self, address: int, size: int, timeout=None):
		"""
		<summary>
		Read the requested number of results from the FPGA result buffer starting at the given address.
		
		Waits until the sequence is finished or a request result transfer is received from the FPGA.
		</summary>
		
		<param name="address" out="false"> <b>Input:</b> The start address of the results to read. </param>
		<param name="size" out="false"> <b>Input:</b> The number results to read. </param>
		<param name="data" out="true" length="size"> <b>Output:</b> The read results. </param>
		
		<returns>Delayed</returns>
		"""
		ticket, event1, event2 = self.post_request(9, struct.pack('<HB', address, size), '<%dL' % (size), True)
		delay = await asyncio.wait_for(event1, timeout=1)
		if timeout is not None and timeout < delay:
			delay = timeout
		val = await asyncio.wait_for(event2, delay + 1)
		return val if len(val) > 1 else val[0]

	async def write(self, address: int, size: int, data: list, timeout=1):
		"""
		<summary>
		Development only function.
		</summary>
		
		<param name="address" out="false"></param>
		<param name="size" out="false"></param>
		<param name="data" out="false" length="size"></param>
		
		<returns>Immediately</returns>
		"""
		ticket, event = self.post_request(10, struct.pack(f'<HH{size}H', address, size, *data), '<', False)
		await asyncio.wait_for(event, timeout)

	async def read(self, address: int, size: int, timeout=1):
		"""
		<summary>
		Development only function.
		</summary>
		
		<param name="address" out="false"></param>
		<param name="size" out="false"></param>
		<param name="data" out="true" length="size"></param>
		
		<returns>Immediately</returns>
		"""
		ticket, event = self.post_request(11, struct.pack('<HH', address, size), '<%dH' % (size), False)
		val = await asyncio.wait_for(event, timeout)
		return val if len(val) > 1 else val[0]

	async def write_special_function_register(self, address: int, data: int, timeout=1):
		"""
		<summary>
		Development only function.
		</summary>
		
		<param name="address" out="false"></param>
		<param name="data" out="false"></param>
		
		<returns>Immediately</returns>
		"""
		ticket, event = self.post_request(12, struct.pack('<BB', address, data), '<', False)
		await asyncio.wait_for(event, timeout)

	async def read_special_function_register(self, address: int, timeout=1):
		"""
		<summary>
		Development only function.
		</summary>
		
		<param name="address" out="false"></param>
		<param name="pData" out="true"></param>
		
		<returns>Immediately</returns>
		"""
		ticket, event = self.post_request(13, struct.pack('<B', address), '<B', False)
		val = await asyncio.wait_for(event, timeout)
		return val if len(val) > 1 else val[0]

	async def send_interlock_event(self, status: int):
		"""
		<summary>
		Send an asynchronous message to the controller when an interlock event is detected.
		</summary>
		
		<param name="status" out="false"> <b>Input:</b> 
		    Bit0 = Output (low if output is on),
		    Bit1 = Error (low if no error detected),
		    Bit2 = Filter Door Sensor (low if filter door closed),
		    Bit3 = Dispenser Sensor (low if dispenser closed)
		</param>
		
		<returns>Immediately</returns>
		"""
		for callback in self.__send_interlock_event_events:
			if inspect.iscoroutinefunction(callback):
				await callback(status)
			else:
				callback(status)

	def subscribe_send_interlock_event(self, callback, no_duplicate=True):
		if no_duplicate and (callback in self.__send_interlock_event_events):
			return 
		self.__send_interlock_event_events.append(callback)

	def unsubscribe_send_interlock_event(self, callback):
		self.__send_interlock_event_events = list(filter(lambda x: x != callback, self.__send_interlock_event_events))

	async def send_error_event(self, status: int):
		"""
		<summary>
		Send an asynchronous message to the controller when an error event is detected.
		</summary>
		
		<param name="status" out="false"> <b>Input:</b> 
		    Bit0 = Alpha Laser Temperature Error,
		    Bit1 = HTS Alpha Laser Temperature Error
		</param>
		
		<returns>Immediately</returns>
		"""
		for callback in self.__send_error_event_events:
			if inspect.iscoroutinefunction(callback):
				await callback(status)
			else:
				callback(status)

	def subscribe_send_error_event(self, callback, no_duplicate=True):
		if no_duplicate and (callback in self.__send_error_event_events):
			return 
		self.__send_error_event_events.append(callback)

	def unsubscribe_send_error_event(self, callback):
		self.__send_error_event_events = list(filter(lambda x: x != callback, self.__send_error_event_events))

